# AIR Avionics Air Control Display ACD-57 simulation by D-ECHO based on

# A3XX Lower ECAM Canvas
# Joshua Davidson (it0uchpods)
#######################################

var ACD57_display = nil;
var ACD57_start = nil;
var ACD57_combi = nil;
var ACD57_xpdr_ctrl = nil;
var ACD57_chnl_sel = nil;
var ACD57_baro = nil;
var ACD57_vol_ctrl = nil;

var state = 0;
var ACD57_volts	= props.globals.getNode("/systems/electrical/outputs/acd-57", 1);

var big_knob_func = props.globals.initNode("/instrumentation/acd57/big-knob-function", "open-vol-ctrl", "STRING");
var sml_knob_func = props.globals.initNode("/instrumentation/acd57/sml-knob-function", "open-vol-ctrl", "STRING");

var page = "combi";
#	combi
#	xpdr_ctrl	Transponder Control
#	chnl_sel	Channel Selection
#	baro		Baro
#	vol_ctrl	Volume Control


#	Transponder Part
var transponder	= props.globals.getNode("/instrumentation/transponder[0]");
var inputs	= transponder.getNode("inputs");

var mode_prop	= inputs.initNode("knob-mode", 0, "INT");
var idcode_prop	= transponder.initNode("id-code", 7000, "INT");
var active_prop	= inputs.initNode("active-change", 0, "INT");

var mode_array	= [ "OFF", "SBY", "TEST", "GND", "ON", "ALT" ];

var internal_code = [ 0, 0, 0, 0 ];

var ident_prop_arr	= [	inputs.initNode("digit[0]", 7, "INT"), 
				inputs.initNode("digit[1]", 0, "INT"), 
				inputs.initNode("digit[2]", 0, "INT"), 
				inputs.initNode("digit[3]", 0, "INT"),	];

#	Altimeter Part
setprop("instrumentation/acd57-altimeter/serviceable", 1 );
var altitude 	= props.globals.getNode("/instrumentation/acd57-altimeter/indicated-altitude-ft", 1);
var qnh_prop	= props.globals.getNode("/instrumentation/acd57-altimeter/setting-hpa", 1);
var previous_press = nil;
var alt_unit	= "m";

#	COMM Part
var comm	= props.globals.getNode("/instrumentation/comm[0]/", 1);
var freq	= comm.getNode("frequencies", 1);
var freq_sel	= freq.getNode("selected-mhz", 1);
var freq_sby	= freq.getNode("standby-mhz", 1);
var vol_act	= comm.getNode("volume", 1);
var vol_sby	= comm.initNode("volume-standby", 0.5, "DOUBLE");

var previous_sby_freq = nil;

var instrument_dir = "Aircraft/miniLAK/Models/Instruments/ACD-57/";


var canvas_ACD57_base = {
	init: func(canvas_group, file) {
		var font_mapper = func(family, weight) {
			return "LiberationFonts/LiberationSans-Bold.ttf";
		};
		
		
		canvas.parsesvg(canvas_group, file, {'font-mapper': font_mapper});
		
		var svg_keys = me.getKeys();
		foreach (var key; svg_keys) {
			me[key] = canvas_group.getElementById(key);
			var clip_el = canvas_group.getElementById(key ~ "_clip");
			if (clip_el != nil) {
				clip_el.setVisible(0);
				var tran_rect = clip_el.getTransformedBounds();
				var clip_rect = sprintf("rect(%d,%d, %d,%d)", 
							tran_rect[1], # 0 ys
							tran_rect[2], # 1 xe
							tran_rect[3], # 2 ye
							tran_rect[0]); #3 xs
							#   coordinates are top,right,bottom,left (ys, xe, ye, xs) ref: l621 of simgear/canvas/CanvasElement.cxx
				me[key].set("clip", clip_rect);
				me[key].set("clip-frame", canvas.Element.PARENT);
			}
		}
		
		me.page = canvas_group;
		
		return me;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
		if( state == 2 ){
			ACD57_start.page.hide();
			if( page == "combi" ){
				ACD57_combi.page.show();
				ACD57_combi.update();
				ACD57_xpdr_ctrl.page.hide();
				ACD57_vol_ctrl.page.hide();
				ACD57_chnl_sel.page.hide();
				ACD57_baro.page.hide();
			} elsif( page == "xpdr_ctrl" ){
				ACD57_combi.page.hide();
				ACD57_xpdr_ctrl.page.show();
				ACD57_vol_ctrl.page.hide();
				ACD57_chnl_sel.page.hide();
				ACD57_baro.page.hide();
			} elsif( page == "vol_ctrl" ){
				ACD57_combi.page.hide();
				ACD57_xpdr_ctrl.page.hide();
				ACD57_vol_ctrl.page.show();
				ACD57_chnl_sel.page.hide();
				ACD57_baro.page.hide();
			} elsif( page == "baro" ){
				ACD57_combi.page.hide();
				ACD57_xpdr_ctrl.page.hide();
				ACD57_vol_ctrl.page.hide();
				ACD57_chnl_sel.page.hide();
				ACD57_baro.page.show();
				ACD57_baro.update();
			} elsif( page = "chnl_sel" ){
				ACD57_combi.page.hide();
				ACD57_xpdr_ctrl.page.hide();
				ACD57_vol_ctrl.page.hide();
				ACD57_chnl_sel.page.show();
				ACD57_baro.page.hide();
			}
		} elsif( state == 1 ){
			ACD57_start.page.show();
			ACD57_combi.page.hide();
			ACD57_xpdr_ctrl.page.hide();
			ACD57_vol_ctrl.page.hide();
			ACD57_chnl_sel.page.hide();
			ACD57_baro.page.hide();
		} else {
			ACD57_start.page.hide();
			ACD57_combi.page.hide();
			ACD57_xpdr_ctrl.page.hide();
			ACD57_vol_ctrl.page.hide();
			ACD57_chnl_sel.page.hide();
			ACD57_baro.page.hide();
		}
	},
};


var canvas_ACD57_combi = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_ACD57_combi , canvas_ACD57_base] };
		m.init(canvas_group, file);
		
		return m;
	},
	getKeys: func() {
		return [ "xpdr.mode", "xpdr.squawk", "qnh", "alt_unit", "alt.100", "alt.rolling_digits", "alt.scale", "comm.act", "comm.sby" ];
	},
	on_xpdr_mode_change: func {
		me["xpdr.mode"].setText( mode_array[ mode_prop.getIntValue() ] );
	},
	on_xpdr_squawk_change: func {
		me["xpdr.squawk"].setText( sprintf("%04d", idcode_prop.getIntValue() ) );
	},
	on_qnh_change: func {
		me["qnh"].setText( sprintf("%5.1f", qnh_prop.getDoubleValue() ) );
	},
	on_act_freq_change: func {
		me["comm.act"].setText( sprintf("%6.3f", freq_sel.getDoubleValue() ) );
	},
	on_sby_freq_change: func {
		me["comm.sby"].setText( sprintf("%6.3f", freq_sby.getDoubleValue() ) );
	},
	update: func {
		# ALT
		if( alt_unit == "m" ){
			var alt = altitude.getDoubleValue() * FT2M;
		} else {
			var alt = altitude.getDoubleValue();
		}
		me["alt.100"].setText(sprintf("%4d", math.floor((alt)/100)));
		me["alt.rolling_digits"].setTranslation(0,math.mod(alt, 100)*7.8);
		me["alt.scale"].setTranslation(0,(math.mod(alt, 50))*2);
		me["alt_unit"].setText( alt_unit );
	},
	
};

var canvas_ACD57_xpdr_ctrl = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_ACD57_xpdr_ctrl , canvas_ACD57_base] };
		m.init(canvas_group, file);
		m.act_digit = 3;
		return m;
	},
	getKeys: func() {
		return [ "squawk.digit.1", "squawk.digit.2", "squawk.digit.3", "squawk.digit.4", "active_digit_indicator", "mode_text" ];
	},
	on_active_digit_change: func ( act_digit ){
		me.act_digit = act_digit;
		for( var i = 1; i <= 4; i += 1 ){
			if( i == ( 4 - me.act_digit ) ){
				me["squawk.digit."~ i ].setColor( 0, 1, 1 );
			} else {
				me["squawk.digit."~ i ].setColor( 0, 0, 0 );
			}
		}
		me["active_digit_indicator"].setTranslation( 88.67 * ( 3 - me.act_digit ), 0 );
	},
	on_digit_change: func {
		me["squawk.digit."~ ( 4 - me.act_digit )].setText( sprintf("%1d", internal_code[ me.act_digit ] ) );
	},
	on_page_open: func {
		me["squawk.digit.1"].setText( sprintf("%1d", internal_code[3] ) );
		me["squawk.digit.2"].setText( sprintf("%1d", internal_code[2] ) );
		me["squawk.digit.3"].setText( sprintf("%1d", internal_code[1] ) );
		me["squawk.digit.4"].setText( sprintf("%1d", internal_code[0] ) );
		
		me["mode_text"].setText( "MODE: "~ mode_array[ mode_prop.getIntValue() ] );
	},
	apply_code: func {
		forindex( var i; ident_prop_arr ){
			ident_prop_arr[i].setIntValue( internal_code[ i ] );
		}
	}
};

var canvas_ACD57_baro = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_ACD57_baro , canvas_ACD57_base] };
		m.init(canvas_group, file);
		
		return m;
	},
	getKeys: func() {
		return ["baro.1","baro.01","result_alt"];
	},
	update: func {
		me["baro.1"].setText( sprintf("%4d", math.floor( qnh_prop.getDoubleValue() ) ) );
		me["baro.01"].setText( sprintf("%1d", math.round( 10 * math.mod( qnh_prop.getDoubleValue(), 1 ) ) ) );
		
		if( alt_unit == "m" ){
			me["result_alt"].setText( sprintf("%4d", math.round( altitude.getDoubleValue() * FT2M ) ) ~ "m" );
		} else {
			me["result_alt"].setText( sprintf("%4d", math.round( altitude.getDoubleValue() ) ) ~ "ft" );
		}
	}
};

var canvas_ACD57_vol_ctrl = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_ACD57_vol_ctrl , canvas_ACD57_base] };
		m.init(canvas_group, file);
		
		return m;
	},
	getKeys: func() {
		return ["freq.act", "freq.sby", "volume.act.bar", "volume.sby.bar", "volume.act.digits", "volume.sby.digits"];
	},
	on_vol_act_change: func {
		var vol = vol_act.getDoubleValue();
		
		me["volume.act.digits"].setText( sprintf("%3d", math.round( 100 * vol ) ) );
		me["volume.act.bar"].setTranslation( -( 1 - vol ) * 359.9, 0 );
		me["volume.act.digits"].setTranslation( -( 1 - vol ) * 359.9, 0 );
	},
	on_vol_sby_change: func {
		var vol = vol_sby.getDoubleValue();
		
		me["volume.sby.digits"].setText( sprintf("%3d", math.round( 100 * vol ) ) );
		me["volume.sby.bar"].setTranslation( -( 1 - vol ) * 359.9, 0 );
		me["volume.sby.digits"].setTranslation( -( 1 - vol ) * 359.9, 0 );
	},
	setup: func {
		me["freq.act"].setText( sprintf("%6.3f", freq_sel.getDoubleValue() ) );
		me["freq.sby"].setText( sprintf("%6.3f", freq_sby.getDoubleValue() ) );
		
		me.on_vol_act_change();
		me.on_vol_sby_change();
	}
};

var canvas_ACD57_chnl_sel = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_ACD57_chnl_sel , canvas_ACD57_base] };
		m.init(canvas_group, file);
		
		return m;
	},
	getKeys: func {
		return [ "freq.mhz", "freq.khz", "channel_name" ];
	},
	on_sby_freq_change: func {
		var freq = freq_sby.getDoubleValue();
		me["freq.mhz"].setText( sprintf("%3d", math.floor( freq ) ) );
		me["freq.khz"].setText( sprintf("%03d", math.round( 1000 * math.mod( freq, 1 ) ) ) );
		
		me["channel_name"].setText( "" );
		findFrequencyID( freq );
	},
	on_find_freq_id: func ( id ){
		# Called from findFrequencyID
		me["channel_name"].setText( id );
	},
};

var findFrequencyID = func( freq ) {
	var id = "";
	
	var closebyAirports = findAirportsWithinRange( 30 );
	foreach( var el; closebyAirports ){
		var comms = airportinfo( el.id ).comms();
		foreach( var x; comms ){
			if( id != "" ) break;
			if( math.abs( x.frequency - freq ) < 0.005 ){
				id = el.name ~ " "~ x.ident;
			}
		}
	}
	
	if( id != "" ){
		ACD57_chnl_sel.on_find_freq_id( id );
	}
}


var canvas_ACD57_start = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_ACD57_start , canvas_ACD57_base] };
		m.init(canvas_group, file);
		
		return m;
	},
	getKeys: func {
		return [];
	},
	update: func() {
	}
};

var on_page_change = func {
	
	if( page == "combi" ){
		big_knob_func.setValue( "open-vol-ctrl" );
		sml_knob_func.setValue( "open-vol-ctrl" );
		
		ACD57_combi.on_xpdr_mode_change();
		ACD57_combi.on_xpdr_squawk_change();
		ACD57_combi.on_qnh_change();
		ACD57_combi.on_act_freq_change();
		ACD57_combi.on_sby_freq_change();
	} elsif( page == "xpdr_ctrl" ){
		big_knob_func.setValue( "xpdr-act-digit" );
		sml_knob_func.setValue( "xpdr-chng-digit" );
		
		# Copy active squawk code to internal variable ("scratchpad")
		forindex( var i; ident_prop_arr ){
			internal_code[ i ] = ident_prop_arr[i].getIntValue();
		}
		
		active_prop.setIntValue( 3 );
		ACD57_xpdr_ctrl.on_page_open();
	} elsif( page == "baro" ){
		previous_press = qnh_prop.getDoubleValue();
		
		big_knob_func.setValue( "hpa" );
		sml_knob_func.setValue( "dapa" );
	} elsif( page == "vol_ctrl" ){
		big_knob_func.setValue( "vol-sby" );
		sml_knob_func.setValue( "vol-act" );
		
		ACD57_vol_ctrl.setup();
	} elsif( page == "chnl_sel" ){
		previous_sby_freq = freq_sby.getDoubleValue();
		
		big_knob_func.setValue( "freq-mhz" );
		sml_knob_func.setValue( "freq-khz" );
		
		ACD57_chnl_sel.on_sby_freq_change();
	}
}
var change_page = func( new_page ){
	page = new_page;
	on_page_change();
}

var check_state = func () {
	if ( state == 0 ){
		if ( ACD57_volts.getDoubleValue() >= 9.0 ) {
			state = 1;
			ACD57_combi.on_xpdr_mode_change();
			ACD57_combi.on_xpdr_squawk_change();
			ACD57_combi.on_qnh_change();
			ACD57_combi.on_act_freq_change();
			ACD57_combi.on_sby_freq_change();
			settimer( func() { state = 2; }, 0.8);
		}
	} else {
		if ( ACD57_volts.getDoubleValue() < 9.0 ){
			state = 0;
		}
	}
}

setlistener(ACD57_volts, func{ check_state(); } );

var softkey_status = [ 1, 1, 1, 1 ];
var softkey = [
	func( i ){
		if( i and softkey_status[0] ){
			softkey_status[0] = 0;
			if( page == "combi" ){
				# "Say Again" feature - currently not implemented
			} elsif( page == "xpdr_ctrl" or page == "vol_ctrl" ){
				change_page( "combi" );
			} elsif( page == "baro" ){
				if( previous_press != nil ){
					qnh_prop.setDoubleValue( previous_press );
				}
				change_page( "combi" );
			} elsif( page == "chnl_sel" ){
				if( previous_sby_freq != nil ){
					freq_sby.setDoubleValue( previous_sby_freq );
				}
				change_page( "combi" );
			}
		} elsif( !i ){
			softkey_status[0] = 1;
		}
	},
	func( i ){
		if( i and softkey_status[1] ){
			softkey_status[1] = 0;
			
			if( page == "combi" ){
				change_page( "chnl_sel" );
			} elsif( page == "xpdr_ctrl" ){
				ident_prop_arr[0].setIntValue( 0 );
				ident_prop_arr[1].setIntValue( 0 );
				ident_prop_arr[2].setIntValue( 0 );
				ident_prop_arr[3].setIntValue( 7 );
				change_page( "combi" );
			}
		} elsif( !i ){
			softkey_status[1] = 1;
		}
	},
	func( i ){
		if( i and softkey_status[2] ){
			softkey_status[2] = 0;
			
			if( page == "combi" ){
				page = "xpdr_ctrl";
				on_page_change();
			} elsif( page == "xpdr_ctrl" ){
				if( mode_prop.getIntValue() != 5 ){
					mode_prop.setIntValue( 5 );
				} else {
					mode_prop.setIntValue( 3 );
				}
				
				ACD57_xpdr_ctrl.apply_code();
				change_page( "combi" );
			} elsif( page == "baro" ){
				if( alt_unit == "m" ) {
					alt_unit = "ft";
				} else {
					alt_unit = "m";
				}
			}
		} elsif( !i ){
			softkey_status[2] = 1;
		}
	},
	func( i ){
		if( i and softkey_status[3] ){
			softkey_status[3] = 0;
			
			if( page == "combi" ){
				change_page( "baro" );
			} elsif( page == "baro" ){
				qnh_prop.setDoubleValue( 1013.25 );
				change_page( "combi" );
			}
		} elsif( !i ){
			softkey_status[3] = 1;
		}
		
	},
];

var swap_comm_freq = func {
	var temp = freq_sel.getDoubleValue();
	freq_sel.setDoubleValue( freq_sby.getDoubleValue() );
	freq_sby.setDoubleValue( temp );
}

var knob_pressed = func( i ){
	if( !i ){ return; }	# possibly more elaborate functions in the future
	if( page == "combi" ){
		swap_comm_freq();
	} elsif( page == "xpdr_ctrl" ){
		ACD57_xpdr_ctrl.apply_code();
		page = "combi";
		on_page_change();
	} elsif( page == "baro" or page == "chnl_sel" ){
		page = "combi";
		on_page_change();
	}
}
var chng_xpdr_digit = func( i ){
	internal_code[ ACD57_xpdr_ctrl.act_digit ] += i;
	ACD57_xpdr_ctrl.on_digit_change();
}

var on_xpdr_mode_change = func {
	if( page == "combi" ){
		ACD57_combi.on_xpdr_mode_change();
	}
}
var on_xpdr_squawk_change = func {
	if( page == "combi" ){
		ACD57_combi.on_xpdr_squawk_change();
	}
}
var on_sby_freq_change = func {
	if( page == "combi" ){
		ACD57_combi.on_sby_freq_change();
	} elsif( page == "chnl_sel" ){
		ACD57_chnl_sel.on_sby_freq_change();
	}
}

var ls = setlistener("sim/signals/fdm-initialized", func {
	ACD57_display = canvas.new({
		"name": "ACD57",
		"size": [644, 484],
		"view": [644, 484],
		"mipmapping": 1,
	});
	ACD57_display.addPlacement({"node": "acd-57.display"});
	
	ACD57_start	= canvas_ACD57_start.new(	ACD57_display.createGroup(), instrument_dir~"acd-57-start.svg");
	ACD57_combi	= canvas_ACD57_combi.new( 	ACD57_display.createGroup(), instrument_dir~"acd-57-combi.svg");
	ACD57_xpdr_ctrl	= canvas_ACD57_xpdr_ctrl.new(	ACD57_display.createGroup(), instrument_dir~"acd-57-xpdr-control.svg");
	ACD57_vol_ctrl	= canvas_ACD57_vol_ctrl.new(	ACD57_display.createGroup(), instrument_dir~"acd-57-volume-control.svg");
	ACD57_chnl_sel	= canvas_ACD57_chnl_sel.new(	ACD57_display.createGroup(), instrument_dir~"acd-57-channel-selection.svg");
	ACD57_baro	= canvas_ACD57_baro.new(	ACD57_display.createGroup(), instrument_dir~"acd-57-baro.svg");
	
	setlistener( mode_prop, on_xpdr_mode_change );
	setlistener( idcode_prop, on_xpdr_squawk_change );
	
	setlistener( qnh_prop, func{ ACD57_combi.on_qnh_change() } );
	
	setlistener( freq_sel, func{ ACD57_combi.on_act_freq_change() } );
	setlistener( freq_sby, on_sby_freq_change );
	
	setlistener( active_prop, func{ ACD57_xpdr_ctrl.on_active_digit_change( active_prop.getIntValue() ) } );
	
	setlistener( vol_act, func{ ACD57_vol_ctrl.on_vol_act_change() } );
	setlistener( vol_sby, func{ ACD57_vol_ctrl.on_vol_sby_change() } );
	
	var base_update = maketimer( 0.2, func {canvas_ACD57_base.update();});
	base_update.start();
	removelistener(ls);
});
