# EXPORT : functions ending by export are called from xml
# CRON : functions ending by cron are called from timer
# SCHEDULE : functions ending by schedule are called from cron
# HUMAN : functions ending by human are called by artificial intelligence



# This file contains checklist tasks.


# ================
# VIRTUAL ENGINEER
# ================

Virtualengineer = {};

Virtualengineer.new = func {
   var obj = { parents : [Virtualengineer,Virtualcrew.new("/systems/engineer")], 

               airbleedsystem : nil,
               autopilotsystem : nil,
               electricalsystem : nil,
               enginesystem : nil,
               fuelsystem : nil,
               hydraulicsystem : nil,
               voicecrew : nil,

               phase : Voicephase.new("/systems/engineer"),
               sensor : Voicesensor.new("/systems/engineer"),
               captain : AsynchronousCrew.new("/systems/engineer"),
               engineer : AsynchronousCrew.new("/systems/engineer"),
               airbleedtask : AirbleedTask.new(),
               electricaltask : ElectricalTask.new(),
               enginetask : EngineerEngineTask.new(),
               fueltask : FuelTask.new(),
               hydraulictask : HydraulicTask.new(),
               icingtask : IcingTask.new(),
               lightingtask : EngineerLightingTask.new(),
               mwstask : EngineerMwsTask.new(),
               navigationtask : EngineerNavigationTask.new(),
               navigation : Navigation.new(),

               FUELSEC : 15.0,
               CRUISESEC : 10.0,
               TAKEOFFSEC : 5.0,
               REHEATSEC : 4.0,

               timerengineer : nil,
               timerengineerslow : nil,
               timerreheat : nil,
               timerweighthuman : nil,
               rates : 0.0,
               
               FOGMETER : 1000,                                  # visibility
               
               FOGDEGC : 3,
               ICEDEGC : 0
         };

    obj.init();

    return obj;
}

Virtualengineer.init = func {
    me.timerengineer = maketimer(me.CRUISESEC, me, me.schedule);
    me.timerengineer.simulatedTime = 1;
    me.timerengineer.singleShot = 1;

    me.timerengineerslow = maketimer(me.FUELSEC, me, me.slowschedule);
    me.timerengineerslow.simulatedTime = 1;
    me.timerengineerslow.singleShot = 1;

    me.timerreheat = maketimer(me.REHEATSEC, me, me.reheatcron);
    me.timerreheat.singleShot = 1;

    me.timerweighthuman = maketimer(me.CRUISESEC, me, me.setweighthuman);
    me.timerweighthuman.simulatedTime = 1;
    me.timerweighthuman.singleShot = 1;
    me.timerweighthuman.start();
}

Virtualengineer.set_relation = func( airbleed, autopilot, electrical, engine, fuel, hydraulic, lighting, voice ) {
    me.airbleedsystem = airbleed;
    me.autopilotsystem = autopilot;
    me.electricalsystem = electrical;
    me.enginesystem = engine;
    me.fuelsystem = fuel;
    me.hydraulicsystem = hydraulic;
    me.voicecrew = voice;

    me.captain.set_relation( autopilot, lighting );
    me.engineer.set_relation( autopilot, lighting );

    me.airbleedtask.set_relation( me, airbleed, voice );
    me.electricaltask.set_relation( me, electrical, voice );
    me.enginetask.set_relation( me, engine, voice );
    me.fueltask.set_relation( me, fuel );
    me.hydraulictask.set_relation( me, hydraulic );
    me.lightingtask.set_relation( me );
    me.icingtask.set_relation( me );
    me.mwstask.set_relation( me );
    me.navigationtask.set_relation( me );
}

Virtualengineer.stateexport = func( targetstate ) {
    me.phase.set_rates( me.CRUISESEC );
    me.phase.schedule();
    me.sensor.schedule();


    me.set_state( constant.TRUE );
    
    if( targetstate == "takeoff" ) {
        me.beforetakeoff();
    }
    
    elsif( targetstate == "taxi" ) {
        me.afterstart();
        me.taxi( constant.FALSE );
    }
    
    elsif( targetstate == "climb" ) {
        me.fueltask.slowschedule();
        me.aftertakeoff();
        me.climb();
    }
    
    elsif( targetstate == "cruise" ) {
        me.fueltask.slowschedule();
        me.aftertakeoff();
        me.climb();
        me.transsonic();
    }
    
    elsif( targetstate == "descent" ) {
        me.fueltask.slowschedule();
        me.aftertakeoff();
        me.climb();
        me.transsonic();
        me.descent();
    }
    
    elsif( targetstate == "approach" ) {
        me.approach();
    }
    
    elsif( targetstate == "final" ) {
        me.beforelanding();
    }
    
    elsif( targetstate == "parking" ) {
        me.afterlanding();
        me.parking();
    }
    
    elsif( targetstate == "stopover" ) {
        me.afterlanding();
        me.parking();
        me.stopover();
    }

    me.set_state( constant.FALSE );
}

Virtualengineer.serviceexport = func {
   var serviceable = me.dependency["crew"].getChild("serviceable").getValue();
 
   me.itself["root"].getChild("serviceable").setValue(serviceable);
   
   if( !serviceable ) {
       me.itself["root-ctrl"].getChild("activ").setValue(constant.FALSE);
   }
}

Virtualengineer.toggleexport = func {
    var launch = constant.FALSE;

    if( me.itself["root"].getChild("serviceable").getValue() ) {
        if( !me.itself["root-ctrl"].getChild("activ").getValue() ) {
            launch = constant.TRUE;
        }
    }
 
    me.itself["root-ctrl"].getChild("activ").setValue(launch);

    if( launch and !me.is_running() ) {
        me.captain.set_task();
        
        # must switch lights again
        me.engineer.set_task();

        me.schedule();
        me.slowschedule();
    }
}

Virtualengineer.radioexport = func() {
    me.engineer.radioexport( constant.TRUE );
}

Virtualengineer.reheatexport = func {
    # at first engine 2 3.
    if( !me.sensor.has_reheat() ) {
        for( var i = constantaero.ENGINE2; i <= constantaero.ENGINE3 ; i = i+1 ) {
             me.dependency["engine-ctrl"][i].getChild("reheat").setValue( constant.TRUE );
        }
 
        me.toggleclick("reheat-2-3");

        # then, engineer sets engines 1 4.
        me.timerreheat.restart(me.REHEATSEC);
    }

    else {
        for( var i = 0; i < constantaero.NBENGINES; i = i+1 ) {
             me.dependency["engine-ctrl"][i].getChild("reheat").setValue( constant.FALSE );
        }
 
        me.toggleclick("reheat-off");
    }
}

Virtualengineer.reheatcron = func {
    if( me.sensor.has_reheat() ) {
        me.dependency["engine-ctrl"][constantaero.ENGINE1].getChild("reheat").setValue( constant.TRUE );
        me.dependency["engine-ctrl"][constantaero.ENGINE4].getChild("reheat").setValue( constant.TRUE );
 
        me.toggleclick("reheat-1-4");
    }
}

Virtualengineer.veryslowschedule = func {
    me.navigation.schedule();
}

Virtualengineer.slowschedule = func {
    me.reset();

    me.rates = me.FUELSEC;

    if( me.itself["root"].getChild("serviceable").getValue() ) {
        me.phase.set_rates( me.rates );
        me.phase.schedule();
        me.sensor.schedule();

        # a slow change is more natural.
        me.fueltask.slowschedule();
        me.lightingtask.navigationlights( constant.TRUE );

        me.timestamp();
    }

    # run slow
    if( me.itself["root-ctrl"].getChild("activ").getValue() ) {
        me.timerengineerslow.restart(me.rates);
    }
    else {
        me.timerengineerslow.stop();
    }
}

Virtualengineer.schedule = func {
    me.reset();

    me.rates = me.CRUISESEC;

    if( me.itself["root"].getChild("serviceable").getValue() ) {
        me.supervisor();
    }
    else {
        me.itself["root"].getChild("activ").setValue(constant.FALSE);
    }

    # run
    if( me.itself["root-ctrl"].getChild("activ").getValue() ) {
        me.set_running();

        me.timerengineer.restart(me.rates);
    }
    else {
        me.timerengineer.stop();
    }
}

Virtualengineer.supervisor = func {
    if( me.itself["root-ctrl"].getChild("activ").getValue() ) {
        if( me.checklist.is_beforetakeoff() or me.checklist.is_approach() or me.checklist.is_aftertakeoff() ) {
            me.rates = me.TAKEOFFSEC;
        }
        
        me.phase.set_rates( me.rates );
        me.phase.schedule();
        me.sensor.schedule();

        me.receive_checklist();
        me.receive_emergency();


        # normal
        if( me.checklist.is_beforetakeoff() ) {
            me.set_activ();
            me.beforetakeoff();
        }

        elsif( me.checklist.is_taxi() ) {
            me.set_activ();
            me.taxi( constant.TRUE );
        }

        elsif( me.checklist.is_afterstart() ) {
            me.set_activ();
            me.afterstart();
        }

        elsif( me.checklist.is_pushback() ) {
            me.set_activ();
            me.pushback();
        }

        elsif( me.checklist.is_enginestart() ) {
            me.set_activ();
            me.enginestart();
        }

        elsif( me.checklist.is_beforestart() ) {
            me.set_activ();
            me.beforestart();
        }

        elsif( me.checklist.is_cockpit() ) {
            me.set_activ();
            me.cockpit();
        }

        elsif( me.checklist.is_preliminary() ) {
            me.set_activ();
            me.preliminary();
        }

        elsif( me.checklist.is_external() ) {
            me.set_activ();
            me.external();
        }

        elsif( me.checklist.is_stopover() ) {
            me.set_activ();
            me.stopover();
        }

        elsif( me.checklist.is_parking() ) {
            me.set_activ();
            me.parking();
        }

        elsif( me.checklist.is_afterlanding() ) {
            me.set_activ();
            me.afterlanding();
        }

        elsif( me.checklist.is_beforelanding() ) {
            me.set_activ();
            me.beforelanding();
        }

        elsif( me.checklist.is_approach() ) {
            me.set_activ();
            me.approach();
        }

        elsif( me.checklist.is_descent() ) {
            me.set_activ();
            me.descent();
        }

        elsif( me.checklist.is_transsonic() ) {
            me.set_activ();
            me.transsonic();
        }

        elsif( me.checklist.is_climb() ) {
            me.set_activ();
            me.climb();
        }

        elsif( me.checklist.is_aftertakeoff() ) {
            me.set_activ();
            me.aftertakeoff();
        }


        # emergency
        elsif( me.checklist.is_fourengineflameout() ) {
            me.set_activ();
            me.fourengineflameout();
        }

        elsif( me.checklist.is_fourengineflameoutmach1() ) {
            me.set_activ();
            me.fourengineflameoutmach1();
        }


        me.allways();

        me.rates = me.randoms( me.rates );

        me.timestamp();
    }

    me.itself["root"].getChild("activ").setValue(me.is_activ());
}

Virtualengineer.get_phase = func {
    return me.phase;
}

Virtualengineer.get_sensor = func {
    return me.sensor;
}


# ------
# FLIGHT
# ------
Virtualengineer.allways = func {
    me.setweight();

    me.engineer.do_task( "engineer", me );
    me.captain.do_task( "captain", me );

    me.navigationtask.allinsready();
}

Virtualengineer.aftertakeoff = func {
    if( !me.phase.is_agl_reheat() or !me.phase.is_speed_approach() ) {
        me.reheatcut();
    }

    if( me.phase.is_agl_climb() ) {
        me.enginetask.enginerating( constantaero.RATINGFLIGHT );

        if( me.has_completed() ) {
            me.voicecrew.engineercheck( "completed" );
            me.voicecrew.nochecklistexport();
        }
    }
}

Virtualengineer.climb = func {
    me.fueltask.takeoffcg( constant.FALSE );

    me.enginetask.setenginecontrol( constantaero.SCHEDULENORMAL );

    if( me.has_completed() ) {
        me.voicecrew.engineercheck( "completed" );
        me.voicecrew.nochecklistexport();
    }
}

Virtualengineer.transsonic = func {
    me.icingtask.staticheater( constant.FALSE );

    me.icingtask.engineantiicing( constant.FALSE );
    me.icingtask.wingantiicing( constant.FALSE );

    if( me.can() ) {
        if( me.phase.is_mach_cruise() ) {
            me.reheatcut();
        }

        # waits
        else {
            me.done_allways();
        }
    }

    if( me.can() ) {
        if( me.phase.is_altitude_cruise() ) {
            me.enginetask.flightrating( constantaero.RATINGCRUISE );
        }

        # waits
        else {
            me.done_allways();
        }
    }

    if( me.has_completed() ) {
        me.voicecrew.engineercheck( "completed" );
        me.voicecrew.cruiseclimbexport();
    }
}

Virtualengineer.descent = func {
    me.enginetask.flightrating( constantaero.RATINGCLIMB );

    me.icingtask.staticheater( constant.TRUE );

    if( me.has_completed() ) {
        me.voicecrew.engineercheck( "completed" );
        me.voicecrew.nochecklistexport();
    }
}

Virtualengineer.approach = func {
    me.enginetask.setenginecontrol( constantaero.SCHEDULEAPPROACH );

    if( me.has_completed() ) {
        me.voicecrew.engineercheck( "completed" );
        me.voicecrew.nochecklistexport();
    }
}

Virtualengineer.beforelanding = func {
    if( me.has_completed() ) {
        me.voicecrew.engineercheck( "completed" );
        me.voicecrew.nochecklistexport();
    }
}


# ------
# GROUND
# ------
Virtualengineer.afterlanding = func {
    me.enginetask.groundidle( constant.TRUE );
    me.enginetask.autoignition( constant.FALSE );

    me.enginetask.taxioutboard();

    # because of engine shutdown
    me.mwstask.cancelaudio();

    me.icingtask.staticheater( constant.FALSE );
    me.icingtask.airdataheater( constant.FALSE );
    me.icingtask.drainmastheater( constant.FALSE );

    me.icingtask.engineantiicing( constant.FALSE );
    me.icingtask.wingantiicing( constant.FALSE );

    if( me.has_completed() ) {
        me.voicecrew.engineercheck( "completed" );
        me.voicecrew.nochecklistexport();
    }
}

Virtualengineer.parking = func {
    me.electricaltask.groundpower( constant.TRUE );

    me.enginetask.throttleidle();

    me.electricaltask.waitgroundpowerbreaker( constant.TRUE );

    # must wait for nose raise (copilot)
    if( !me.sensor.is_nose_up() ) {
        me.done();
    }

    me.enginetask.shutengines();
    me.enginetask.throttlemaster( constant.TRUE );

    me.lightingtask.anticollision( constant.FALSE );
    me.icingtask.engineantiicing( constant.FALSE );
    me.enginetask.autoignition( constant.FALSE );

    me.airbleedtask.shutairbleeds();
    me.airbleedtask.airconditioning( constant.TRUE );

    me.electricaltask.battery( constant.FALSE );

    me.checklist.ins( me, constantaero.INS3, constantaero.INSALIGN );

    if( me.has_completed() ) {
        me.voicecrew.engineercheck( "completed" );
        me.voicecrew.nochecklistexport();
    }
}

Virtualengineer.stopover = func {
    if( me.checklist.is_startup() ) {
        me.enginetask.autoignition( constant.FALSE );
        me.enginetask.shutengines();

        if( me.can() ) {
            me.checklist.unset_startup();
        }
    }

    me.navigationtask.adc( constant.FALSE );
    me.navigationtask.allins( constantaero.INSOFF );

    me.electricaltask.groundpowerbreaker( constant.FALSE );

    if( me.has_completed() ) {
        me.voicecrew.engineercheck( "completed" );
        me.voicecrew.nochecklistexport();
    }
}

Virtualengineer.external = func {
    # check availability of connections
    me.electricaltask.groundpower( constant.FALSE );
    me.airbleedtask.airconditioning( constant.FALSE );
    me.airbleedtask.airbleed();

    if( me.has_completed() ) {
        me.voicecrew.nochecklistexport();
    }
}

Virtualengineer.preliminary = func {
    me.electricaltask.waitgroundpowerbreaker( constant.TRUE );

    # because of electrical power
    me.mwstask.cancelaudio();

    if( me.dependency["temperature"].getValue() < me.ICEDEGC ) {
        me.icingtask.drainmastheater( constant.TRUE );
    }

    me.checklist.ins( me, constantaero.INS1, constantaero.INSALIGN );
    me.checklist.ins( me, constantaero.INS3, constantaero.INSALIGN );
    me.checklist.ins( me, constantaero.INS2, constantaero.INSALIGN );

    me.navigationtask.adc( constant.TRUE );

    if( me.has_completed() ) {
        me.voicecrew.nochecklistexport();
    }
}

Virtualengineer.cockpit = func {
    me.enginetask.flightrating( constantaero.RATINGCLIMB );
    me.enginetask.throttlemaster( constant.TRUE );
    me.enginetask.autoignition( constant.FALSE );
    me.enginetask.enginerating( constantaero.RATINGTAKEOFF );

    me.checklist.ins( me, constantaero.INS1, constantaero.INSALIGN );
    me.checklist.ins( me, constantaero.INS3, constantaero.INSALIGN );
    me.checklist.ins( me, constantaero.INS2, constantaero.INSALIGN );

    me.navigationtask.compass();

    me.airbleedtask.crossbleedvalve( constantaero.ENGINE1, constant.FALSE );
    me.airbleedtask.crossbleedvalve( constantaero.ENGINE2, constant.TRUE );
    me.airbleedtask.crossbleedvalve( constantaero.ENGINE3, constant.TRUE );
    me.airbleedtask.crossbleedvalve( constantaero.ENGINE4, constant.FALSE );

    me.fueltask.fuelconsumed();

    if( me.has_completed() ) {
        me.voicecrew.nochecklistexport();
    }
}

Virtualengineer.beforestart = func {
    me.enginetask.throttleidle();
    me.lightingtask.navigationlights( constant.TRUE );
    me.enginetask.throttlemaster( constant.FALSE );

    me.electricaltask.battery( constant.TRUE );
    me.lightingtask.anticollision( constant.TRUE );

    if( me.has_completed() ) {
        if( !me.wait_ground() ) {
            me.voicecrew.engineercheck( "clearance" );

            # must wait for ground answer
            me.done_ground();
        }
    
        else {
            me.voicecrew.engineercheck( "clear" );

            me.reset_ground();
            me.voicecrew.nochecklistexport();
        }
    }
}

Virtualengineer.enginestart = func {
    me.startengine( constantaero.ENGINE3 );
    me.startengine( constantaero.ENGINE4 );

    me.startengine( constantaero.ENGINE2 );
    me.startengine( constantaero.ENGINE1 );

    if( me.has_completed() ) {
        me.voicecrew.startedexport();
    }
}

Virtualengineer.pushback = func {
    me.startengine( constantaero.ENGINE3 );
    me.startengine( constantaero.ENGINE2 );

    me.airbleedtask.crossbleedvalve( constantaero.ENGINE2, constant.TRUE );
    me.airbleedtask.crossbleedvalve( constantaero.ENGINE3, constant.TRUE );

    if( me.dependency["tractor"].getChild("engine14").getValue() ) {
        me.airbleedtask.crossbleedvalve( constantaero.ENGINE4, constant.TRUE );
        me.startengine( constantaero.ENGINE4 );

        me.airbleedtask.crossbleedvalve( constantaero.ENGINE1, constant.TRUE );
        me.startengine( constantaero.ENGINE1 );

        me.completed();
    }

    # waits for pushback
    else {
        me.done();
    }

    if( me.has_completed() ) {
        me.voicecrew.engineercheck( "completed" );
        me.voicecrew.startedexport();
    }
}

Virtualengineer.afterstart = func {
    me.enginetask.groundidle( constant.TRUE );

    if( me.noinstrument["temperature"].getValue() < me.FOGDEGC and
        me.noinstrument["visibility"].getValue() < me.FOGMETER ) {
        me.icingtask.engineantiicing( constant.TRUE );
    }

    # should do it
    me.electricaltask.waitgroundpowerbreaker( constant.FALSE );

    if( me.can() ) {
        if( !me.checklist.is_completed() ) {
            me.completed();
        }
    }
}

Virtualengineer.taxi = func( callout ) {
    me.enginetask.enginerating( constantaero.RATINGTAKEOFF );
    me.enginetask.autoignition( constant.TRUE );

    me.icingtask.drainmastheater( constant.TRUE );

    me.enginetask.flightrating( constantaero.RATINGCLIMB );

    me.icingtask.staticheater( constant.TRUE );
    me.icingtask.airdataheater( constant.FALSE );

    # normal or flyover at takeoff
    me.enginetask.setenginecontrol( constantaero.SCHEDULENORMAL );
    
    me.fueltask.takeoffcg( constant.TRUE );

    if( callout ) {
        if( me.has_completed() ) {
            me.voicecrew.engineercheck( "completed" );
            me.voicecrew.runwayexport();
        }
    }
}

Virtualengineer.beforetakeoff = func {
    # not ready at FG launch
    me.taxi( constant.FALSE );

    me.enginetask.groundidle( constant.FALSE );

    if( me.has_completed() ) {
        me.voicecrew.engineercheck( "completed" );
        me.voicecrew.nochecklistexport();
    }
}


# ---------
# EMERGENCY
# ---------
Virtualengineer.fourengineflameout = func {
    if( !me.sensor.is_allengines() ) {
        # easier to work
        me.mwstask.cancelaudio();

        me.hydraulictask.unguardrat( constantaero.AP2 );
        me.deployrat( constantaero.AP2 );

        me.enginetask.throttleidle();
        me.enginetask.autoignition( constant.FALSE );
        me.enginetask.throttlemaster( constant.FALSE );
        me.enginetask.relightallengines( constant.TRUE );

        me.emergencyenginestart( constantaero.ENGINE2 );
        me.emergencyenginestart( constantaero.ENGINE4 );
        me.emergencyenginestart( constantaero.ENGINE3 );
        me.emergencyenginestart( constantaero.ENGINE1 );
    }

    # restores situation, before emergency
    else {
        me.electricaltask.emergencyrelight( -1 );
        me.enginetask.relightallengines( constant.FALSE );
        me.enginetask.autoignition( constant.TRUE );
    }

    if( me.has_completed() ) {
        me.voicecrew.noemergencyexport();
    }
}

Virtualengineer.fourengineflameoutmach1 = func {
    if( !me.sensor.is_allengines() ) {
        # easier to work
        me.mwstask.cancelaudio();

        me.enginetask.throttleidle();
        me.enginetask.autoignition( constant.FALSE );
        me.enginetask.relightallengines( constant.TRUE );

        me.emergencyenginestart( constantaero.ENGINE2 );
        me.emergencyenginestart( constantaero.ENGINE4 );
        me.emergencyenginestart( constantaero.ENGINE3 );
        me.emergencyenginestart( constantaero.ENGINE1 );
    }

    # restores situation, before emergency
    else {
        me.electricaltask.emergencyrelight( -1 );
        me.enginetask.relightallengines( constant.FALSE );
        me.enginetask.autoignition( constant.TRUE );
    }

    if( me.has_completed() ) {
        me.voicecrew.noemergencyexport();
    }
}


# ------
# ENGINE
# ------
Virtualengineer.reheatcut = func {
    if( me.can() ) {
        if( me.sensor.has_reheat() ) {
            me.reheatexport();
        }
    }
}

Virtualengineer.startengine = func( index ) {
    me.airbleedtask.crossbleedvalve( index, constant.TRUE );

    if( me.can() ) {
        if( !me.dependency["engine"][index].getChild("running").getValue() ) {
            me.enginetask.enginestarter( index );
            me.enginetask.hpvalve( index, constant.TRUE );
        }

        # waits for shut of start valve
        if( me.can() )  {
           if( me.dependency["engine-ctrl"][index].getChild("starter").getValue() ) {
               me.done();
           }

           else {
               me.airbleedtask.bleedvalve( index, constant.TRUE );
           }
        }
    }
}

Virtualengineer.emergencyenginestart = func( index ) {
    if( me.can() ) {
        if( !me.dependency["engine"][index].getChild("running").getValue() ) {
            var set = me.electricalsystem.enginerelightexport( index );

            me.electricaltask.emergencyrelight( set );
            me.enginetask.hpvalve( index, constant.TRUE );

            # waits for engine running
            if( me.can() ) {
                me.done();
            }
        }
    }
}


# ----
# FUEL
# ----
# set weight datum
Virtualengineer.setweight = func {
    if( me.can() ) {

        # after fuel loading
        var reset = me.dependency["fuel-sys"].getChild("reset").getValue();

        # after 1 reset of fuel consumed
        if( !reset ) {
            for( var i = 0; i < constantaero.NBENGINES; i = i+1 ) {
                 if( me.dependency["fuel-consumed"][i].getChild("reset").getValue() ) {
                     reset = constant.TRUE;
                     break;
                 }
            }
        }

        if( reset ) {
            me.setweighthuman();
            me.toggleclick("set-weight");
        }
    }
}

Virtualengineer.setweighthuman = func {
    var totallb = me.dependency["fuel"].getChild("total-lbs").getValue();

    me.fuelsystem.setweighthuman( totallb * constant.LBTOKG );
}


# ===========
# ENGINE TASK
# ===========

EngineerEngineTask = {};

EngineerEngineTask.new = func {
   var obj = { parents : [EngineerEngineTask,VirtualTask.new(),System.new("/systems/engineer")],

               enginesystem : nil,
               voicecrew : nil
         };

   obj.init();

   return obj;
};

EngineerEngineTask.init = func {
}

EngineerEngineTask.set_relation = func( crew, engine, voice ) {
    me.crewhuman = crew;
    me.enginesystem = engine;
    me.voicecrew = voice;
}

EngineerEngineTask.taxioutboard = func {
    # optional in checklist
    if( me.dependency["crew-ctrl"].getChild("stop-engine23").getValue() ) {
        for( i = constantaero.ENGINE2; i <= constantaero.ENGINE3; i=i+1 ) {
             if( me.can() ) {
                 # taxi with outboard engines
                 me.hpvalve( i, constant.FALSE );
             }
         }
    }
    else {
        me.voicecrew.terminalexport();
    }
}

EngineerEngineTask.groundidle = func( set ) {
    if( me.can() ) {
        var path = "";

        path = "ground-idle14";

        if( me.dependency["engine-all"].getChild(path).getValue() != set ) {
            me.dependency["engine-all"].getChild(path).setValue( set );
        }

        path = "ground-idle23";

        if( me.dependency["engine-all"].getChild(path).getValue() != set ) {
            me.dependency["engine-all"].getChild(path).setValue( set );
        }
    }
}

EngineerEngineTask.hpvalve = func( index, set ) {
    if( me.can() ) {
        if( me.dependency["engine-ctrl"][index].getChild("hp-valve").getValue() != set ) {
            me.dependency["engine-ctrl"][index].getChild("hp-valve").setValue( set );
            me.toggleclick("hp-valve-" ~ index);
        }
    }
}

EngineerEngineTask.autoignition = func( set ) {
    if( me.can() ) {
        for( var i = 0; i < constantaero.NBENGINES; i = i + 1 ) {
             if( me.dependency["engine-ctrl"][i].getChild("autoignition").getValue() != set ) {
                 me.dependency["engine-ctrl"][i].getChild("autoignition").setValue( set );
                 me.toggleclick("autoignition-" ~ i);
                 break;
             }
        }
    }
}

EngineerEngineTask.enginestarter = func( index ) {
    if( me.can() ) {
        if( !me.dependency["engine-ctrl"][index].getChild("starter").getValue() ) {
            me.enginesystem.starter( index );
            me.toggleclick("starter-" ~ index);
        }
    }
}

EngineerEngineTask.flightrating = func( flight ) {
    if( me.can() ) {
        var flightnow = "";

        for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
             flightnow = me.dependency["engine-ctrl"][i].getChild("rating-flight").getValue();
             if( flightnow != flight ) {
                 me.dependency["engine-ctrl"][i].getChild("rating-flight").setValue(flight);
                 me.toggleclick("rating-" ~ i ~ "-" ~ flight);
                 break;
             }
        }
    }
}

EngineerEngineTask.enginerating = func( rating ) {
    if( me.can() ) {
        var ratingnow = "";

        for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
             ratingnow = me.dependency["engine-ctrl"][i].getChild("rating").getValue();
             if( ratingnow != rating and rating == constantaero.RATINGFLIGHT ) {
                 if( !me.dependency["gear-ctrl"].getChild("gear-down").getValue() ) {
                     me.dependency["engine-ctrl"][i].getChild("rating").setValue(rating);
                     me.toggleclick("rating-" ~ i ~ "-" ~ rating);
                     break;
                 }
             }
        }
    }
}

EngineerEngineTask.setenginecontrol = func( set ) {
    if( me.can() ) {
        if( !me.dependency["engine-all"].getChild("schedule-auto").getValue() ) {
            me.dependency["engine-all"].getChild("schedule-auto").setValue( constant.TRUE );
            me.toggleclick("engine-sched-auto");
        }

        elsif( me.dependency["engine-all"].getChild("schedule").getValue() != set ) {
            me.dependency["engine-all"].getChild("schedule").setValue( set );
            me.toggleclick("engine-schedule");
        }
    }
}

EngineerEngineTask.shutengines = func {
    if( me.can() ) {
        for( var i = 0; i < constantaero.NBENGINES; i = i + 1 ) {
             me.hpvalve( i, constant.FALSE );
        }
    }
}

EngineerEngineTask.relightallengines = func( set ) {
    if( me.can() ) {
        for( var i = 0; i < constantaero.NBENGINES; i = i + 1 ) {
             me.relightengine( i, set );
        }
    }
}

EngineerEngineTask.relightengine = func( index, set ) {
    if( me.can() ) {
        if( me.dependency["engine-sys"][index].getChild("relight").getValue() != set ) {
            me.enginesystem.relight( index, set );
            me.toggleclick("relight-" ~ index);
        }
    }
}

EngineerEngineTask.throttleidle = func {
    if( me.can() ) {
        var found = constant.FALSE;

        for( var i = 0; i < constantaero.NBENGINES; i = i + 1 ) {
             if( me.dependency["engine-ctrl"][i].getChild("throttle-manual").getValue() != constantaero.THROTTLEIDLE ) {
                 me.enginesystem.set_throttle( constantaero.THROTTLEIDLE );
                 found = constant.TRUE;
             }
        }

        # the 4 levers at once
        if( found ) {
            me.done("idle");
        }
    }
}

EngineerEngineTask.throttlemaster = func( set ) {
    if( me.can() ) {
        for( var i = 0; i < constantaero.NBENGINES; i = i + 1 ) {
             if( me.dependency["throttle"][i].getChild("off").getValue() != set ) {
                 me.dependency["throttle"][i].getChild("off").setValue( set );
                 me.toggleclick("throttle-master-" ~ i);
                 break;
             }
        }
    }
}


# ===============
# NAVIGATION TASK
# ===============

EngineerNavigationTask = {};

EngineerNavigationTask.new = func {
   var obj = { parents : [EngineerNavigationTask,VirtualTask.new(),System.new("/systems/engineer")]
         };

   obj.init();

   return obj;
};

EngineerNavigationTask.init = func {
}

EngineerNavigationTask.allins = func( mode ) {
    for( var i = 0; i < constantaero.NBINS; i = i + 1 ) {
         me.get_checklist().ins( me, i, mode );
    }
}

EngineerNavigationTask.allinsready = func {
    for( var i = 0; i < constantaero.NBINS; i = i + 1 ) {
         me.insready( i );
    }
}

EngineerNavigationTask.insready = func( index ) {
    if( me.can() ) {
        if( me.dependency["ins"][index].getNode("msu").getChild("aligned").getValue() ) {
            var INSNAV = 1;

            if( me.dependency["ins"][index].getNode("msu").getChild("mode").getValue() != INSNAV ) {
                # highest quality
                if( me.dependency["ins"][index].getNode("msu/status[4]").getValue() == 1 ) {
                    me.dependency["ins"][index].getNode("msu").getChild("mode").setValue( INSNAV );
                    me.toggleclick("ins-ready-" ~ index);
                }
            }
        }
    }
}

EngineerNavigationTask.compass = func {
    for( var i = 0; i < constantaero.NBAUTOPILOTS; i = i + 1 ) {
         if( me.can() ) {
             if( me.dependency["compass"][i].getChild("mode-dg").getValue() ) {
                 me.dependency["compass"][i].getChild("mode-dg").setValue( constant.FALSE );
                 me.toggleclick("mode-dg-" ~ i);
             }
         }
    }
}

EngineerNavigationTask.adc = func( set ) {
    for( var i = 0; i < constantaero.NBAUTOPILOTS; i = i + 1 ) {
         if( me.can() ) {
             if( me.dependency["adc"][i].getChild("switch").getValue() != set ) {
                 me.dependency["adc"][i].getChild("switch").setValue( set );
                 me.toggleclick("adc-" ~ i);
             }
         }
    }
}


# ==========
# ICING TASK
# ==========

IcingTask = {};

IcingTask.new = func {
   var obj = { parents : [IcingTask,VirtualTask.new(),System.new("/systems/engineer")]
         };

   obj.init();

   return obj;
};

IcingTask.init = func {
}

IcingTask.airdataheater = func( set ) {
}

IcingTask.drainmastheater = func( set ) {
    var path = "";

    for( var i = 0; i < constantaero.NBINS; i = i+1 ) {
         if( me.can() ) {
             path = "mast/heater[" ~ i ~ "]";

             if( me.dependency["anti-icing"].getNode(path).getValue() != set ) {
                 me.dependency["anti-icing"].getNode(path).setValue( set );
                 me.toggleclick("icing-mast-" ~ i);
                 break;
             }
         }
    }
}

IcingTask.staticheater = func( set ) {
    var path = "";

    for( var i = 0; i < constantaero.NBAUTOPILOTS; i = i+1 ) {
         if( me.can() ) {
             path = "static/heater[" ~ i ~ "]";

             if( me.dependency["anti-icing"].getNode(path).getValue() != set ) {
                 if( set and me.get_phase().is_mach_supersonic() ) {
                     # waits
                     me.done_allways();
                 }

                 else {
                     me.dependency["anti-icing"].getNode(path).setValue( set );
                     me.toggleclick("icing-static-" ~ i);
                 }

                 break;
             }
         }
    }
}

IcingTask.engineantiicing = func( set ) {
    var child = nil;

    for( var i = 0; i < constantaero.NBENGINES; i = i+1 ) {
         if( me.can() ) {
             child = me.dependency["anti-icing"].getChild("engine",i);

             if( child.getChild("inlet-vane").getValue() != set ) {
                 child.getChild("inlet-vane").setValue( set );
                 me.toggleclick("icing-engine-" ~ i);
                 break;
             }
         }
    }
}

IcingTask.wingantiicing = func( set ) {
    var path = "";

    if( me.can() ) {
        path = "wing/main-selector";

        if( me.dependency["anti-icing"].getNode(path).getValue() != set ) {
            me.dependency["anti-icing"].getNode(path).setValue( set );
            me.toggleclick("icing-main-" ~ i);
        }
    }

    if( me.can() ) {
        path = "wing/alternate-selector";

        if( me.dependency["anti-icing"].getNode(path).getValue() != set ) {
            me.dependency["anti-icing"].getNode(path).setValue( set );
            me.toggleclick("icing-alt-" ~ i);
        }
    }
}


# =============
# LIGHTING TASK
# =============

EngineerLightingTask = {};

EngineerLightingTask.new = func {
   var obj = { parents : [EngineerLightingTask,VirtualTask.new(),System.new("/systems/engineer")]
         };

   obj.init();

   return obj;
};

EngineerLightingTask.init = func {
}

EngineerLightingTask.anticollision = func( set ) {
    if( me.can() ) {
        if( me.dependency["lighting"].getChild("strobe").getValue() != set ) {
            me.dependency["lighting"].getChild("strobe").setValue( set );
            me.toggleclick("anti-collision");
        }
    }
}

EngineerLightingTask.navigationlights = func( set ) {
    # optional in checklist
    set = me.is_light_required( set );

    if( me.can() ) {
        if( me.dependency["lighting"].getChild("nav-lights").getValue() != set ) {
            me.dependency["lighting"].getChild("nav-lights").setValue( set );
            me.toggleclick("nav-lights");
        }
    }
}

EngineerLightingTask.is_light_required = func( set ) {
    var result = set;

    if( !me.dependency["landing-lights"].getChild("navigation").getValue() ) {
        result = constant.FALSE;
    }
    
    elsif( me.is_state() ) {
        # optional
        result = constant.FALSE;
    }
    
    elsif( me.dependency["landing-lights"].getChild("as-required").getValue() ) {
        if( set ) {
            if( !constant.is_lighting( me.noinstrument["sun"].getValue(), me.noinstrument["altitude"].getValue() ) and
                me.noinstrument["visibility"].getValue() > constant.HAZEMETER ) {
                result = constant.FALSE;
            }
        }
    }
    
    return result;
}


# ========
# MWS TASK
# ========

EngineerMwsTask = {};

EngineerMwsTask.new = func {
   var obj = { parents : [EngineerMwsTask,VirtualTask.new(),System.new("/systems/engineer")]
         };

   obj.init();

   return obj;
};

EngineerMwsTask.init = func {
}

EngineerMwsTask.cancelaudio = func {
    if( me.can() ) {
        if( !me.dependency["audio"].getChild("cancel").getValue() ) {
            me.dependency["audio"].getChild("cancel").setValue(constant.TRUE);
            me.toggleclick("cancel-audio");
        }
    }
}


# ===============
# ELECTRICAL TASK
# ===============

ElectricalTask = {};

ElectricalTask.new = func {
   var obj = { parents : [ElectricalTask,VirtualTask.new(),System.new("/systems/engineer")],
   
               electricalsystem : nil,
               voicecrew : nil
         };

   obj.init();

   return obj;
};

ElectricalTask.init = func {
}

ElectricalTask.set_relation = func( crew, electrical, voice ) {
    me.crewhuman = crew;
    me.electricalsystem = electrical;
    me.voicecrew = voice;
}

ElectricalTask.groundpower = func( call ) {
    if( me.can() ) {
        if( !me.electricalsystem.has_ground() ) {
            if( !me.wait_ground() and call ) {
                 me.voicecrew.engineercheck( "ground power" );

                 # must wait for electrical system run (ground)
                 me.done_ground();
            }

            else  {
                 me.electricalsystem.groundserviceexport();

                 me.reset_ground();
                 me.done();
            }
        }

        elsif( !me.dependency["gear-ctrl"].getChild("wheel-chocks").getValue() ) {
            if( !me.wait_ground() and call ) {
                # must wait for wheel chocks (ground)
                me.done_ground();
            }

            else  {
                me.dependency["gear-ctrl"].getChild("wheel-chocks").setValue( constant.TRUE );

                me.reset_ground();
                me.done();
           }
        }
    }
}

ElectricalTask.waitgroundpowerbreaker = func( value ) {
    if( me.electricalsystem.has_ground() == value ) {
        me.groundpowerbreaker( value );
    }

    # waits for electrical power
    else {
        me.done();
    }
}

ElectricalTask.groundpowerbreaker = func( value ) {
    if( me.can() ) {
        if( me.dependency["electric-ac"].getChild("gpb").getValue() != value ) {
            me.dependency["electric-ac"].getChild("gpb").setValue( value );
            me.toggleclick("ground-power");
        }
    }
}

ElectricalTask.battery = func( set ) {
    if( me.can() ) {
        for( var i = 0; i < constantaero.NBAUTOPILOTS; i = i + 1 ) {
             if( me.dependency["electric-dc"].getChild("master-bat",i).getValue() != set ) {
                 me.dependency["electric-dc"].getChild("master-bat",i).setValue( set );
                 me.toggleclick("battery-" ~ i);
                 break;
             }
        }
    }
}

ElectricalTask.emergencyrelight = func( set ) {
    if( me.can() ) {
        if( me.dependency["electric-ac"].getNode("emergency").getChild("relight-selector").getValue() != set ) {
            me.dependency["electric-ac"].getNode("emergency").getChild("relight-selector").setValue( set );
            me.electricalsystem.emergencyrelightexport();
            me.toggleclick("emerg-relight-" ~ set);
        }
    }
}


# ==============
# AIR BLEED TASK
# ==============

AirbleedTask = {};

AirbleedTask.new = func {
   var obj = { parents : [AirbleedTask,VirtualTask.new(),System.new("/systems/engineer")],
   
               airbleedsystem : nil,
               voicecrew : nil
         };

   obj.init();

   return obj;
};

AirbleedTask.init = func {
}

AirbleedTask.set_relation = func( crew, hydraulic, voice ) {
    me.crewhuman = crew;
    me.airbleedsystem = hydraulic;
    me.voicecrew = voice;
}

AirbleedTask.airconditioning = func( call ) {
    if( me.can() ) {
        if( !me.airbleedsystem.has_reargroundservice() ) {
            if( !me.wait_ground() and call ) {
                me.voicecrew.engineercheck( "air conditioning" );

                # must wait for temperature system run (ground)
                me.done_ground();
            }

            else  {
                 me.airbleedsystem.reargroundserviceexport();

                 me.reset_ground();
                 me.done();
            }
        }
    }
}

AirbleedTask.airbleed = func {
    if( me.can() ) {
        if( !me.airbleedsystem.has_groundservice() ) {
            me.airbleedsystem.groundserviceexport();

            me.done();
        }
    }
}

AirbleedTask.shutairbleeds = func {
    if( me.can() ) {
        for( var i = 0; i < constantaero.NBENGINES; i = i + 1 ) {
             if( me.can() ) {
                 me.bleedvalve( i, constant.FALSE );
             }
        }
    }
}

AirbleedTask.bleedvalve = func( index, set ) {
    if( me.can() ) {
        if( me.dependency["air-bleed"][index].getChild("bleed-valve").getValue() != set ) {
            me.dependency["air-bleed"][index].getChild("bleed-valve").setValue( set );
            me.toggleclick("bleed-valve-" ~ index);
        }
    }
}

AirbleedTask.crossbleedvalve = func( index, set ) {
    if( me.can() ) {
        if( me.dependency["air-bleed"][index].getChild("cross-bleed-valve").getValue() != set ) {
            me.dependency["air-bleed"][index].getChild("cross-bleed-valve").setValue( set );
            me.toggleclick("cross-bleed-valve-" ~ index);
        }
    }
}


# ==============
# HYDRAULIC TASK
# ==============

HydraulicTask = {};

HydraulicTask.new = func {
   var obj = { parents : [HydraulicTask,VirtualTask.new(),System.new("/systems/engineer")],
   
               hydraulicsystem : nil
         };

   obj.init();

   return obj;
};

HydraulicTask.init = func {
}

HydraulicTask.set_relation = func( crew, hydraulic ) {
    me.crewhuman = crew;
    me.hydraulicsystem = hydraulic;
}

HydraulicTask.unguardrat = func( index ) {
    if( me.can() ) {
        if( me.dependency["rat"][index].getChild("guard").getValue() ) {
            me.dependency["rat"][index].getChild("guard").setValue( constant.FALSE );
            me.toggleclick("rat-guard-" ~ index);
        }
    }
}

HydraulicTask.deployrat = func( index ) {
    if( me.can() ) {
        if( !me.dependency["rat"][index].getChild("on").getValue() ) {
            me.dependency["rat"][index].getChild("on").setValue( constant.TRUE );
            me.hydraulicsystem.ratdeployexport();
            me.toggleclick("rat-" ~ index);
        }
    }
}


# =========
# FUEL TASK
# =========

FuelTask = {};

FuelTask.new = func {
   var obj = { parents : [FuelTask,VirtualTask.new(),System.new("/systems/engineer")],
   
               fuelsystem : nil,
 
               RESETKG : 0,

               MAXTOPERCENT : 54.0,                              # maximum at takeoff
               MAXPERCENT : 53.6,                                # maximum on ground
               CGPERCENT : 0.25                                  # checklist
         };

   obj.init();

   return obj;
};

FuelTask.init = func {
}

FuelTask.set_relation = func( crew, fuel ) {
    me.crewhuman = crew;
    me.fuelsystem = fuel;
}

FuelTask.slowschedule = func {
    me.fuelcg();
    me.fuelcross();
}

FuelTask.fuelcg = func {
    if( me.can() ) {
        var engine = constant.FALSE;
        var forward = constant.FALSE;
        var aft = constant.FALSE;
        var afttrim = constant.FALSE;
        var max = 0.0;
        var min = 0.0;
        var cg = 0.0;
        var mean = 0.0;
        var offset = 0.0;

        if( me.dependency["cg"].getChild("serviceable").getValue() ) {
            max = me.dependency["cg"].getChild("max-percent").getValue();
            min = me.dependency["cg"].getChild("min-percent").getValue();
            cg = me.dependency["cg"].getChild("percent").getValue();

            engine = constant.FALSE;
            afttrim = constant.FALSE;

            # emergency
            if( cg < min ) {
                me.log("below-min");
                aft = constant.TRUE;
                engine = constant.TRUE;
                afttrim = constant.TRUE;
            }
            elsif( cg > max ) {
                me.log("above-max");
                forward = constant.TRUE;
                engine = constant.TRUE;
            }

            # above 250 kt beyond 10000 ft
            elsif( me.noinstrument["altitude"].getValue() > constantaero.APPROACHFT ) {
                # anticipates aft shift
                if( me.get_phase().is_climb_fast() and me.get_phase().is_mach_climb() ) {
                    mean = min + ( max - min ) / ( 3 / 4 );
                }

                # anticipates forwards shift
                elsif( me.get_phase().is_descent_fast() and me.get_phase().is_mach_climb() ) {
                    mean = min + ( max - min ) / 4;
                }

                # cruise
                else {
                    mean = min + ( max - min ) / 2;
                }

                if( cg < mean - me.CGPERCENT and cg < max - me.CGPERCENT ) {
                    me.log("aft");
                    aft = constant.TRUE;
                    engine = constant.TRUE;
                }

                # don't move on ground, if within limits
                elsif( cg > mean + me.CGPERCENT and cg > min + me.CGPERCENT and cg > me.MAXPERCENT ) {
                    me.log("forward");
                    forward = constant.TRUE;
                    engine = constant.TRUE;
                }
           }

           me.applyfuelcg( forward, aft, engine, afttrim );
       }


       if( me.itself["root"].getNode("cg/aft").getValue() != aft or
           me.itself["root"].getNode("cg/forward").getValue() != forward ) {
           me.itself["root"].getNode("cg/aft").setValue(aft);
           me.itself["root"].getNode("cg/forward").setValue(forward);

           me.toggleclick();
       }
    }
}

FuelTask.fuelcross = func {
    if( me.can() ) {
        var cross = me.fuelsystem.lowlevel();

        if( me.itself["root"].getNode("cg/cross").getValue() != cross ) {
            me.itself["root"].getNode("cg/cross").setValue(cross);
            me.fuelsystem.togglecrossfeed( cross );

            me.toggleclick("cross-feed");
        }
    }
}

FuelTask.applyfuelcg = func( forward, aft, engine, afttrim) {
    var pump6 = constant.FALSE;
    var pump8 = constant.FALSE;
    var auxilliary = constant.FALSE;

    # pumps
    var empty5 = me.fuelsystem.empty("5");
    var empty5A = me.fuelsystem.empty("5A");
    var empty6 = me.fuelsystem.empty("6");
    var empty7 = me.fuelsystem.empty("7");
    var empty7A = me.fuelsystem.empty("7A");
    var empty8 = me.fuelsystem.empty("8");
    var empty9 = me.fuelsystem.empty("9");
    var empty10 = me.fuelsystem.empty("10");
    var empty11 = me.fuelsystem.empty("11");

    # no 2D panel
    me.fuelsystem.aft2Dhuman( constant.FALSE );


    # shut all unused pumps
    me.fuelsystem.pumphuman( "5", !empty5 );
    me.fuelsystem.pumphuman( "7", !empty7 );
    me.fuelsystem.pumphuman( "5A", constant.FALSE );
    me.fuelsystem.pumphuman( "7A", constant.FALSE );

    if( empty5 and !empty6 ) {
        pump6 = constant.TRUE;
    }
    me.fuelsystem.pumphuman( "6", pump6 );

    if( empty7 and !empty8 ) {
        pump8 = constant.TRUE;
    }
    me.fuelsystem.pumphuman( "8", pump8 );

    # engineer normally uses auto trim
    me.fuelsystem.pumphuman( "9", constant.FALSE );
    me.fuelsystem.pumphuman( "10", constant.FALSE );
    me.fuelsystem.pumphuman( "11", constant.FALSE );

    me.fuelsystem.shutstandbyhuman();


    # aft trim
    me.fuelsystem.afttrimhuman( afttrim );


    # transfers auxilliary tanks
    if( empty5 and empty6 ) {
        if( !empty5A ) {
            auxilliary = constant.TRUE;
            me.fuelsystem.transvalvehuman( "5A", constant.TRUE );
            me.fuelsystem.pumphuman( "5A", constant.TRUE );
        }
    }
    if( empty7 and empty8 ) {
        if( !empty7A ) {
            auxilliary = constant.TRUE;
            me.fuelsystem.transvalvehuman( "7A", constant.TRUE );
            me.fuelsystem.pumphuman( "7A", constant.TRUE );
        }
    }
    if( auxilliary ) {
        me.log("auxilliary");
    }


    # low level (emergency)
    if( !auxilliary and me.fuelsystem.lowlevel() ) {
        me.fuelsystem.offautohuman();

        # avoid aft CG  
        if( ( forward or !aft ) and !empty11 ) {
            me.log("low-level");
            me.fuelsystem.pumphuman( "11", constant.TRUE );
            me.fuelsystem.enginehuman( constant.TRUE );
            me.fuelsystem.forwardhuman( constant.TRUE );
        }
        elsif( !empty9 or !empty10 ) {
            me.log("low-level");
            me.fuelsystem.pumphuman( "9", !empty9 );
            me.fuelsystem.pumphuman( "10", !empty10 );
            me.fuelsystem.enginehuman( constant.TRUE );
            me.fuelsystem.afthuman( constant.TRUE );
        }
        # last fuel
        elsif( !empty11 ) {
            me.log("low-level");
            me.fuelsystem.pumphuman( "11", constant.TRUE );
            me.fuelsystem.enginehuman( constant.TRUE );
            me.fuelsystem.forwardhuman( constant.TRUE );
        }
        else {
            me.fuelsystem.enginehuman( constant.FALSE );
        }
    }

    # aft transfert
    elsif( aft ) {
        me.fuelsystem.forwardautohuman( constant.FALSE );
    }

    # forward transfert
    elsif( forward ) {
        me.fuelsystem.forwardautohuman( constant.TRUE );
    }

    # no transfert
    else {
        me.fuelsystem.offautohuman();
        me.fuelsystem.enginehuman( constant.FALSE );
    }
}

FuelTask.fuelconsumed = func {
    if( me.can() ) {
        for( var i = 0; i < constantaero.NBENGINES; i = i + 1 ) {
             if( me.can() ) {
                 me.resetfuelconsumed( i );
             }
        }
    }
}

FuelTask.resetfuelconsumed = func( index ) {
    if( me.can() ) {
        if( me.dependency["fuel-consumed"][index].getChild("total-kg").getValue() > me.RESETKG ) {
            if( !me.dependency["fuel-consumed"][index].getChild("reset").getValue() ) {
                me.dependency["fuel-consumed"][index].getChild("reset").setValue( constant.TRUE );
                me.toggleclick("fuel-consumed-" ~ index);
            }
        }
    }
}

FuelTask.takeoffcg = func( target ) {
    if( me.can() ) {
        var set = constant.FALSE;
        var cgpercent = me.dependency["cg"].getChild("percent").getValue();
        
        if( cgpercent > me.MAXPERCENT and cgpercent <= me.MAXTOPERCENT ) {
            set = target;
        }
        
        if( me.dependency["cg"].getChild("max-performance-to").getValue() != set ) {
            me.dependency["cg"].getChild("max-performance-to").setValue( set );
            me.toggleclick("max-perf-to");
        }
    }
}


# ==========
# NAVIGATION
# ==========

Navigation = {};

Navigation.new = func {
   var obj = { parents : [Navigation,System.new("/systems/engineer")], 
   
               launchhour : 0,
               launchminute : 0,
               launchsecond : 0,

               altitudeft : 0.0,
               lastft : 0.0,

               last : constant.FALSE,

               NOSPEEDFPM : 0.0,

               SUBSONICKT : 480,                                 # estimated ground speed
               FLIGHTKT : 150,                                   # minimum ground speed

               groundkt : 0,

               SUBSONICKGPH : 20000,                             # subsonic consumption

               kgph : 0,

               NOFUELKG : -999,

               totalkg : 0
         };

   obj.init();

   return obj;
}

Navigation.init = func {
   me.launchhour = me.noinstrument["time-real"].getChild("hour").getValue();
   me.launchminute = me.noinstrument["time-real"].getChild("minute").getValue();
   me.launchsecond = me.noinstrument["time-real"].getChild("second").getValue();
}

Navigation.schedule = func {
   me.waypoints();
   me.time();
   me.time_real();
   me.climbrate();
}

Navigation.climbrate = func() {
   var climbfpm = 0;
   var currentft = 0.0;
   var stepft = 0.0;
   
   currentft = me.dependency["altimeter"].getChild("indicated-altitude-ft").getValue();
   stepft = currentft - me.lastft;

   climbfpm = math.round( stepft / 10 ) * 10;
   me.lastft = currentft;

   me.itself["root"].getNode("navigation").getChild("climb-fpm").setValue(climbfpm);
}

Navigation.time = func {
   var elapsedsec = me.dependency["time"].getValue();
   
   var elapsedhours = me.second_to_hour( elapsedsec );
   me.itself["root"].getNode("navigation").getChild("elapsed-hours").setValue(elapsedhours);
}

Navigation.time_real = func {
   var nowhour = me.noinstrument["time-real"].getChild("hour").getValue();
   var nowminute = me.noinstrument["time-real"].getChild("minute").getValue();
   var nowsecond = me.noinstrument["time-real"].getChild("second").getValue();
   
   var elapsedsec = 0;
   
   # change of day
   if( nowhour < me.launchhour ) {
       elapsedsec = (constant.HOURLAST - me.launchhour) * constant.HOURTOSECOND + elapsedsec;
       elapsedsec = (constant.MINUTELAST - me.launchminute) * constant.MINUTETOSECOND + elapsedsec;
       elapsedsec = (constant.SECONDLAST - me.launchsecond) + elapsedsec;

       elapsedsec = nowhour * constant.HOURTOSECOND + elapsedsec;
       elapsedsec = nowminute * constant.MINUTETOSECOND + elapsedsec;
       elapsedsec = nowsecond + elapsedsec;
   }
   else {
       elapsedsec = (nowhour - me.launchhour) * constant.HOURTOSECOND + elapsedsec;
       elapsedsec = (nowminute - me.launchminute) * constant.MINUTETOSECOND + elapsedsec;
       elapsedsec = (nowsecond - me.launchsecond) + elapsedsec;
   }

   var clockstring = me.second_to_time( elapsedsec );
   me.itself["root"].getNode("navigation").getChild("clock-string").setValue(clockstring);
}

Navigation.second_to_hour = func( elapsedsec ) {
   var elapsedhours = elapsedsec / constant.HOURTOSECOND;
   
   elapsedhours = math.round( elapsedhours * 10 ) / 10;
   
   return elapsedhours;
}

Navigation.second_to_time = func( elapsedsec ) {
   var clockhours = int( elapsedsec / constant.HOURTOSECOND );

   var clockminutes = int( (elapsedsec - clockhours * constant.HOURTOSECOND) / constant.MINUTETOSECOND );

   var clockseconds = elapsedsec - clockhours * constant.HOURTOSECOND -  clockminutes * constant.MINUTETOSECOND;
   
   clockstring = sprintf("%02d",clockhours) ~ ":" ~ sprintf("%02d",clockminutes) ~ ":" ~ sprintf("%02d",clockseconds);
   
   return clockstring;
}

Navigation.waypoints = func {
   var groundfps = me.dependency["ins"][2].getNode("computed").getChild("ground-speed-fps").getValue();
   var id = "";
   var distnm = 0.0;
   var targetft = 0;
   var selectft = 0.0;
   var fuelkg = 0.0;
   var speedfpm = 0.0;
   var display = constant.FALSE;

   if( groundfps != nil ) {
       me.groundkt = groundfps * constant.FPSTOKT;
   }

   me.totalkg = me.dependency["fuel"].getChild("total-lbs").getValue() * constant.LBTOKG;

   # on ground
   if( me.groundkt < me.FLIGHTKT ) {
       me.groundkt = me.SUBSONICKT;
       me.kgph = me.SUBSONICKGPH;
   }
   else {
       # gauge is NOT REAL
       me.kgph = me.dependency["fuel"].getNode("fuel-flow-kg_ph").getValue();
   }

   me.altitudeft = me.noinstrument["altitude"].getValue();
   selectft = me.dependency["autoflight"].getChild("altitude-select").getValue();
   me.last = constant.FALSE;


   # waypoint
   for( var i = 2; i >= 0; i = i-1 ) {
        if( i < 2 ) {
            id = me.dependency["waypoint"][i].getChild("id").getValue();
            distnm = me.dependency["waypoint"][i].getChild("dist").getValue();
            targetft = selectft;
        }

        # last
        else {
            id = me.dependency["route-manager"].getNode("wp-last/id",constant.DELAYEDNODE).getValue(); 
            distnm = me.dependency["route-manager"].getNode("wp-last/dist",constant.DELAYEDNODE).getValue(); 
        }

        fuelkg = me.estimatefuelkg( id, distnm );
        speedfpm = me.estimatespeedfpm( id, distnm, targetft );
        
        display = constant.TRUE;
        if( fuelkg == me.NOFUELKG ) {
            display = constant.FALSE;
        }
        me.itself["waypoint"][i].getChild("fuel").setValue(display);
        
        display = constant.TRUE;
        if( speedfpm == me.NOSPEEDFPM ) {
            display = constant.FALSE;
        }
        me.itself["waypoint"][i].getChild("speed").setValue(display);

        # display for FDM debug, or navigation
        me.itself["waypoint"][i].getChild("fuel-kg").setValue(int(math.round(fuelkg)));
        me.itself["waypoint"][i].getChild("speed-fpm").setValue(int(math.round(speedfpm)));
   }
}

Navigation.estimatespeedfpm = func( id, distnm, targetft ) {
   var speedfpm = me.NOSPEEDFPM;
   var minutes = 0.0;

   if( id != "" and distnm != nil ) {
       # last waypoint at sea level
       if( !me.last ) {
           targetft = me.itself["root-ctrl"].getChild("destination-ft").getValue();
           me.last = constant.TRUE;
       }

       minutes = ( distnm / me.groundkt ) * constant.HOURTOMINUTE;
       speedfpm = ( targetft - me.altitudeft ) / minutes;
   }

   return speedfpm;
}

Navigation.estimatefuelkg = func( id, distnm ) {
   var fuelkg = me.NOFUELKG;
   var ratio = 0.0;

   if( id != "" and distnm != nil ) {
       ratio = distnm / me.groundkt;
       fuelkg = me.kgph * ratio;
       fuelkg = me.totalkg - fuelkg;
       if( fuelkg < 0 ) {
           fuelkg = 0;
       }
   }

   return fuelkg;
}
