# Song 120 ELECTRICAL SYSTEM
# Bea Wolf (D-ECHO) 2022

# based on turboprop engine electrical system by Syd Adams and C172S electrical system   ####

#	Components:
#		Engine:			Rotex Electric REX30
#		Motor Controller:	HBC Series V7 280120-3
#			close: https://www.elektrisch-fliegen.de/wp-content/uploads/wpforo/default_attachments/1635762190-HBC_manual_B15_LVMV_100313-B-ENG.pdf
#		Battery:		Battery Management System (BMS): BMS16 (https://www.elektro-ul.de/bms-16.html)
#					2 Batteries (ref. POH, p.30), total capacity 65 Ah at 115 V (ref. POH, p.29)
#					Battery Cells: Sony Sony US18650NC1
#						2900 mAh, 3.6 V, max. discharge current 8A ( ref. http://www.altra.net.pl/pdf/Sony%20US18650NC1.pdf, p.5 )
#						Battery test: https://www.akkuline.de/test/sony-us18650nc1-test-messung
#							max. voltage 4.163 V
#							avg. voltage 3.16 V
#							actual capacity 2758 mAh (95.1%)
#							actual enery: 8.73 Wh (83.62%)
#					assume: 32 cells ( = 2 battery packs * 16 cells each ) -> to reach 115 V at 3.6 V rated voltage
#							actual capacity: 88.256 Ah
#							actual voltage: 133.22 V (max), 101.12 (avg)
#	
#	Ref. POH, p.30: Avionics are supplied by an additional 12V glider battery, which is *not charged* by the main battery
#			suggested avionics battery: LiFePO4, 5 - 15 Ah
#			assume: https://shop.segelflugbedarf24.de/Flugzeugausstattung/Akkus-Energieversorgung/Airbatt-LiFePO4-12V-12Ah-Avionik::2715.html

# Basic props.nas objects
var electrical = props.globals.getNode("systems/electrical");
var electrical_sw = electrical.initNode("internal-switches");
var output = electrical.getNode("outputs");
var breakers = props.globals.initNode("/controls/circuit-breakers");
var controls = props.globals.getNode("/controls/electric");

# Helper functions
var check_or_create = func ( prop, value, type ) {
	var obj = props.globals.getNode(prop, 1);
	if( obj.getValue() == nil ){
		return props.globals.initNode(prop, value, type);
	} else {
		return obj;
	}
}

var comm_ptt = props.globals.getNode("/instrumentation/comm[0]/ptt", 1);

#	Switches
var switches = {
	battery:		controls.initNode("battery-switch",		0,	"BOOL"),
	avionics:		controls.initNode("avionics-switch",		0,	"BOOL"),
};


var eng_power	= props.globals.getNode("/fdm/jsbsim/propulsion/engine/electrical-power-hp", 1);

#	Additional (not directly consumer-related) Circuit Breakers
var circuit_breakers = {
};

#	Internal (virtual) switches
var int_switches = {
};

var delta_sec	=	props.globals.getNode("sim/time/delta-sec");

var BatteryClass = {
	new : func( switch, volt, amps, amp_hours, charge, charge_amps, n){
		m = { 
			parents : [BatteryClass],
			switch:		switch,
			serviceable:	electrical.initNode("/systems/electrical/battery["~n~"]/serviceable", 1, "BOOL"),
			temp:		electrical.initNode("battery["~n~"]/temperature", 15.0, "DOUBLE"),
			rated_volts:	volt,
			ideal_amps:	amps,
			volt_p:		electrical.initNode("battery["~n~"]/volts", 0.0, "DOUBLE"),
			amp_hours:	amp_hours,
			charge:		charge, 
			charge_amps:	charge_amps,
		};
		return m;
	},
	apply_load : func( load ) {
		var dt = delta_sec.getDoubleValue();
		if( me.switch.getBoolValue() ){
			var amphrs_used = load * dt / 3600.0;
			var norm_used = amphrs_used / me.amp_hours;
			me.charge -= norm_used;
			if ( me.charge < 0.0 ) {
				me.charge = 0.0;
			} elsif ( me.charge > 1.0 ) {
				me.charge = 1.0;
			}
			var output =me.amp_hours * me.charge;
			return output;
		}else return 0;
	},
	# model voltage output based on this graph:	https://www.akkuline.de/test/sony-us18650nc1-test-messung
	get_output_volts : func {
		if( me.switch.getBoolValue() ){
			var x = me.charge;
			var factor = 0.0;
			if( x < 0.173 ){
				factor = 0.678 - 0.678 * math.pow( 2.71828, ( - 35 * x ) );
			} elsif( x > 0.927 ) {
				factor = 0.84 - ( 0.84 - 1.157 ) * math.pow( 2.71828, ( -35 * ( 1 - x ) ) );
			} else {
				factor = 0.678 + ( x - 0.173 / ( 0.927 - 0.173 ) ) * ( 0.84 - 0.678 );
			}
			var output = me.rated_volts * factor;
			me.volt_p.setDoubleValue( output );
			return output;
		}else return 0;
	},
	
	get_output_amps : func {
		if( me.switch.getBoolValue() ){
			var x = 1.0 - me.charge;
			var tmp = -(3.0 * x - 1.0);
			var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
			var output =me.ideal_amps * factor;
			return output;
		}else return 0;
	}
};

var battery_main = BatteryClass.new(switches.battery, 115.0, 100, 88.256, 1.0, 15.0, 0);
var battery_12v = BatteryClass.new(switches.avionics, 12.0, 20, 12.0, 1.0, 10.0, 0);

#	Buses
#		Battery Engine Bus (120V)
#		Auxiliary Bus (12V)
#		

#	Consumer Class
#		Functions:
#			* power: takes bus_volts, applies to relevant outputs/ property and returns electrical load
#			* automatically trips its circuit breaker when maximum load is exceeded
var consumer = {
	new: func( name, switch, watts, cb_max ){
		m = { parents : [consumer] };
		m.cb = breakers.initNode(name, 1, "BOOL");
		m.switch_type = "none";
		if( switch != nil ){
			m.switch = switch;
			if ( switch.getType() == "DOUBLE" or switch.getType() == "FLOAT" ) {
				m.switch_type = "double";
			} else if ( switch.getType() == "BOOL" ) {
				m.switch_type = "bool";
			} else {
				die("Consumer (non-int) switch of unsupported type: "~ switch.getType() ~ "!");
			}
		} else {
			m.switch = nil;
		}
		m.output = output.initNode(name, 0.0, "DOUBLE");
		m.watts = watts;
		m.cb_max = cb_max;
		return m;
	},
	power: func( bus_volts ){
		if( me.cb.getBoolValue() and bus_volts != 0.0 ){
			if ( me.switch_type == "none" or ( me.switch_type == "bool" and me.switch.getBoolValue() ) ) {
				me.output.setDoubleValue( bus_volts );
				# if( me.watts/bus_volts > me.cb_max ){
				# 	me.cb.setBoolValue( 0 );
				# 	return 0.0;
				# }
				return me.watts / bus_volts;
			} else if ( me.switch_type == "double" ) {
				me.output.setDoubleValue( bus_volts * me.switch.getDoubleValue() );
				# if( me.watts / bus_volts * me.switch.getDoubleValue() > me.cb_max ){
				# 	me.cb.setBoolValue( 0 );
				# 	return 0.0;
				# }
				return me.watts / bus_volts * me.switch.getDoubleValue();
			} else {
				me.output.setDoubleValue( 0.0 );
				return 0.0;
			}
		} else {
			me.output.setDoubleValue( 0.0 );
			return 0.0;
		}
	},
};
# Consumer with support for integer switches
var consumer_int = {
	new: func( name, switch, watts, int, mode ){
		m = { parents : [consumer_int] };
		m.cb = breakers.initNode(name, 1, "BOOL");
		if ( switch.getType() == "INT" ) {
			m.switch = switch;
			m.int = int;
			# Mode: 0 means "=="; 1 means "!="
			if( mode != nil ){
				m.mode = mode;
			} else {
				m.mode = 0;
			}
		} else {
			die("Consumer (int) switch of unsupported type: "~ switch.getType() ~ "!");
		}
		m.output = output.initNode(name, 0.0, "DOUBLE");
		m.watts = watts;
		return m;
	},
	power: func( bus_volts ){
		if( me.cb.getBoolValue() and bus_volts != 0.0 ){
			if ( ( ( me.mode == 0 and me.switch.getIntValue() == me.int ) or ( me.mode == 1 and me.switch.getIntValue() != me.int ) ) ) {
				me.output.setDoubleValue( bus_volts );
				return me.watts / bus_volts;
			} else {
				me.output.setDoubleValue( 0.0 );
				return 0.0;
			}
		} else {
			me.output.setDoubleValue( 0.0 );
			return 0.0;
		}
	},
};

#	Electrical Bus Class
var bus = {
	new: func( name, on_update, consumers ) {
		m = {	
			parents: [bus],
			name: name,
			volts: check_or_create("systems/electrical/" ~ name ~ "/volts", 0.0, "DOUBLE"),
			amps: check_or_create("systems/electrical/"~ name ~"/amps", 0.0, "DOUBLE"),
			serviceable: check_or_create("systems/electrical/" ~ name ~ "/serviceable", 1, "BOOL"),
			on_update: on_update,
			bus_volts: 0.0,
			consumers: consumers,
			extra_load: 0.0,
			src: "",
		};
		return m;
	},
	update_consumers: func () {
		#print("Update consumers of bus "~ me.name);
		load = 0.0;
		foreach( var c; me.consumers ) {
			load += c.power( me.bus_volts );
		}
		return load;
	},
};
var battery_bus = bus.new(
	"battery-bus",
	func() {
		me.src = "";
		if( me.serviceable.getBoolValue() ){
			me.bus_volts = battery_main.get_output_volts();
		} else {
			me.bus_volts = 0.0;
		}
		
		var load = me.update_consumers();
		load += engine_bus.on_update( me.bus_volts );
		load += engine_avionics_bus.on_update( me.bus_volts );
		
		battery_main.apply_load( load );
		
		me.extra_load = 0.0;
		me.volts.setDoubleValue( me.bus_volts );
		me.amps.setDoubleValue( load );
	},
	[
	],
);

#	Engine Bus
var engine_bus = bus.new(
	"engine-bus",
	func( bv ){
		me.src = "";
		me.bus_volts = bv;
		
		var load = me.update_consumers();
		
		if( me.bus_volts > 0.0 ){
			# Calculate engine load
			var power_kw = math.max( eng_power.getDoubleValue() * 0.7457, 0 );
			# P = U * I -> I = P / U
			load += ( power_kw * 1000 ) / me.bus_volts;
		}
		
		me.volts.setDoubleValue( me.bus_volts );
		me.amps.setDoubleValue( load );
		
		return load;
	},
	[
	],
);

#	Engine Avionics Bus
var engine_avionics_bus = bus.new(
	"engine-avionics-bus",
	func( bv ){
		me.src = "";
		me.bus_volts = bv * 0.1;
		
		var load = me.update_consumers();
		
		me.volts.setDoubleValue( me.bus_volts );
		me.amps.setDoubleValue( load );
		
		return load * 0.1 ;
	},
	[
		consumer.new( "eng_display", nil, 0.5, 999 ),
	],
);


#	Avionic Bus
#		powers:
#			own consumers

#	Salus energy consumption (ref. https://downloads.lxnavigation.com/downloads/manuals/Salus/LX-Salus_users_manual_1_4.pdf, p. 4): 140mA at 12VDC
#	flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf
var avionic_bus = bus.new(
	"avionic-bus",
	func() {
		if( me.serviceable.getBoolValue() ){
			me.bus_volts = battery_12v.get_output_volts();
		} else {
			me.bus_volts = 0.0;
		}
		
		var load = me.update_consumers();
		
		me.volts.setDoubleValue( me.bus_volts );
		
		if( me.bus_volts > 0.0 ){
			battery_12v.apply_load( load );
		}
	},
	[
		consumer.new( "salus", nil, 1.68, 5 ),
		consumer.new( "flarm", nil, 0.66, 5 ),
	],
);

var update_electrical = func {
	
	battery_bus.on_update();
	avionic_bus.on_update();
	
}

var electrical_updater = maketimer( 0.0, update_electrical );
electrical_updater.simulatedTime = 1;
electrical_updater.start();
